import java.util.Iterator;

public class DoublyLinkedList<T> implements Iterable<T> {

    private class Node {
        public T item;
        public Node previous;
        public Node next;

        public Node(T item, Node previous, Node next) {
            this.item = item;
            this.previous = previous;
            this.next = next;
        }
    }

    private Node first, last;
    private int size;

    public int size() {
        return size;
    }

    public void add(T item) {
        Node newNode = new Node(item, null, null);

        if (first == null) { // erster einzufügender Knoten
            first = newNode;
            last = newNode;
        } else {
            last.next = newNode;
            newNode.previous = last;
            last = newNode;
        }
        size++;
    }

    public void addAll(T[] items) {
        for (T item : items) {
            add(item);
        }
    }

    public boolean remove(T item) {
        Node node = find(item);

        if (node == null)
            return false;

        if (first == node) // Anfang löschen, first korrigieren
            first = node.next;

        if (last == node) // Ende löschen, last korrigieren
            last = node.previous;

        if (node.previous != null) // Vorgängerverweis korrigieren
            node.previous.next = node.next;

        if (node.next != null) // Nachfolgerverweis korrigieren
            node.next.previous = node.previous;

        size--;

        return true;
    }

    public boolean insertAfter(T previousItem, T item) {
        Node previousNode = find(previousItem);

        if (previousNode == null)
            return false;

        var newNode = new Node(item, previousNode, previousNode.next);

        previousNode.next = newNode;

        if (newNode.next != null)
            newNode.next.previous = newNode;
        else
            last = newNode;

        size++;

        return true;
    }

    public boolean contains(T item) {
        Node node = find(item);

        return (node != null);
    }

    public void clear() {
        first = null;
        last = null;
        size = 0;
    }

    public T get(int index) {

        Node node = findByIndex(index);
        return node.item;
    }

    public void set(int index, T item) {
        Node node = findByIndex(index);
        node.item = item;
    }

    private Node find(T item) {
        Node node = first;

        while (node != null) {
            if (node.item.equals(item)) {
                return node;
            }

            node = node.next;
        }

        return null;
    }

    private Node findByIndex(int index) {

        if (index >= size)
            throw new IndexOutOfBoundsException();

        Node node = first;
        int i = 0;

        while (node != null) {

            if (i == index)
                return node;

            node = node.next;
            i++;
        }
        return null;
    }

    @Override
    public String toString() {
        String s = "";
        Node node = first;

        while (node != null) {
            s += node.item.toString() + " -> ";
            node = node.next;
        }

        s += " | ";
        node = last;
        while (node != null) {
            s += node.item.toString() + " <- ";
            node = node.previous;
        }

        s += "Count: " + size;

        return s;
    }

    @Override
    public Iterator<T> iterator() {
        return new LinkedListIterator();
    }

    private class LinkedListIterator implements Iterator<T> {
        private Node currentNode = first;

        public boolean hasNext() {
            return (currentNode != null);
        }

        public T next() {
            if (!hasNext()) {
                throw new IndexOutOfBoundsException();
            }

            T item = currentNode.item;
            currentNode = currentNode.next;

            return item;
        }
    }
}