import java.io.*;
import java.net.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

public class HttpServer {

    private ServerSocket listener = null;
    private int port;
    private String localPath;

    public HttpServer(int port, String localPath) {
        this.port = port;
        this.localPath = localPath;
    }

    public int port() {
        return port;
    }

    public String localPath() {
        return localPath;
    }

    public void start() {
        try {
            listener = new ServerSocket(port, 0, InetAddress.getLoopbackAddress());

            Requests requests = new Requests();

            while (true) {
                Socket c = listener.accept();
                requests.add(c);
                if (requests.countRequests() > 1) {
                    requests.process(localPath);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private class Requests {

        private Queue<Socket> queue = new Queue<>();

        public void add(Socket c) {
            queue.enqueue(c);
        }

        public int countRequests() {
            return queue.size();
        }

        public void process(String path) {
            System.out.println("Queue: " + queue.size());

            while (queue.size() != 0) {
                try {
                    Request r = new Request(queue.dequeue(), path);
                    System.out.println("  " + r.getLocalUrl());
                } catch (Exception e) {
                    System.out.println("Error: " + e.getMessage());
                }
            }
        }
    }

    private class Request {
        private Socket c;
        private String localPath;

        private String localUrl;
        private String verb;
        private String httpVersion;
        private List<String> header;

        public String getLocalUrl() {
            return localUrl;
        }

        public String verb() {
            return verb;
        }

        public String httpVersion() {
            return httpVersion;
        }

        public List<String> header() {
            return header;
        }

        public Request(Socket c, String localPath) throws IOException {
            this.c = c;
            this.localPath = localPath;
            this.header = new ArrayList<String>();
            process();
        }

        private void process() throws IOException {

            BufferedReader r = new BufferedReader(new InputStreamReader(c.getInputStream())); // Lesen vom Remote-Host
            BufferedWriter w = new BufferedWriter(new OutputStreamWriter(c.getOutputStream())); // Schreiben an
                                                                                                // Remote-Host

            try {
                handleRequest(r, w);
            } catch (Exception e) {
                failure(w);
                throw e;
            } finally {
                w.close();
                r.close();
                c.close();
            }
        }

        private void handleRequest(BufferedReader r, BufferedWriter w) throws IOException {
            String request = r.readLine();

            if (request != null) {
                String[] tokens = request.split(" ");

                verb = tokens[0].toUpperCase();
                localUrl = tokens[1];
                httpVersion = tokens[2];

                readHeader(r);

                Path file = Paths.get(localPath, localUrl.substring(1));

                String text = new String(Files.readAllBytes(file));

                success(w, getFilenameExtension(localUrl));

                w.write(text);
            }
        }

        private void readHeader(BufferedReader r) throws IOException {
            String attribute;

            while ((attribute = r.readLine()) != null) {
                if (attribute.equals(""))
                    break;
                header.add(attribute);
            }
        }

        private static void success(BufferedWriter stream, String ext) throws IOException {
            stream.write("HTTP/1.0 200 OK\r\n");
            stream.write("Content-Type: text/" + ext + "\r\n");
            stream.write("Connection: close\r\n");
            stream.write("\r\n");
        }

        private static void failure(BufferedWriter stream) throws IOException {
            stream.write("HTTP/1.0 404 not found\r\n");
            stream.write("Connection: close\r\n");
            stream.write("\r\n");
        }

        private static String getFilenameExtension(String filename) {
            var ext = "";

            int index = filename.lastIndexOf('.');

            if (index > 0)
                ext = filename.substring(index + 1);

            return ext;
        }
    }
}