import java.util.ArrayList;

public class Hashtable<K, V> {

    private class Pair {
        private K key;
        private V value;

        public Pair(K key, V value) {
            this.key = key;
            this.value = value;
        }

        public K Key() {
            return key;
        }

        public V Value() {
            return value;
        }

        public void Value(V value) {
            this.value = value;
        }
    }

    private int size;

    public int size() {
        return size;
    }

    private ArrayList<Pair>[] items;

    public double loadFactor() {
        int buckets = 0;

        for (int i = 0; i < items.length; i++) {
            if (items[i] != null) {
                buckets++;
            }
        }

        return 1.0 - ((double) (items.length - buckets) / items.length);
    }

    public double occupationFactor() {
        int buckets = 0;

        for (int i = 0; i < items.length; i++) {
            if (items[i] != null) {
                buckets++;
            }
        }
        return (double) size / buckets;
    }

    @SuppressWarnings("unchecked")
    public Hashtable(int length) {
        length = calcPrimeLength(length);
        items = new ArrayList[length];
    }

    public Hashtable() {
        this(10);
    }

    public void add(K key, V value) {
        if (containsKey(key))
            throw new IllegalArgumentException("Item already exists");

        int hash = getHash(key);

        var list = items[hash];

        if (list == null) {
            list = new ArrayList<Pair>();
            items[hash] = list;
        }

        list.add(new Pair(key, value));

        size++;
    }

    public void remove(K key) {
        int hash = getHash(key);

        ArrayList<Pair> list = items[hash];

        if (list != null) {
            for (int i = 0; i < list.size(); i++) {
                if (list.get(i).Key().equals(key)) {
                    list.remove(i);
                    size--;
                    break;
                }
            }
        }
    }

    public V get(K key) {
        int hash = getHash(key);

        ArrayList<Pair> list = items[hash];

        if (list != null) {
            for (int i = 0; i < list.size(); i++) {
                if (list.get(i).Key().equals(key)) {
                    return list.get(i).Value();
                }
            }
        }
        return null;
    }

    public void set(K key, V value) {
        if (!containsKey(key)) {
            add(key, value);
        } else {
            update(key, value);
        }
    }

    @SuppressWarnings("unchecked")
    public void clear() {
        items = new ArrayList[items.length];
        size = 0;
    }

    public boolean contains(K key) {
        return containsKey(key);
    }

    public boolean containsKey(K key) {
        int hash = getHash(key);

        ArrayList<Pair> list = items[hash];

        if (list != null) {
            for (int i = 0; i < list.size(); i++) {
                if (list.get(i).Key().equals(key)) {
                    return true;
                }
            }
        }
        return false;
    }

    public ArrayList<V> values() {
        ArrayList<V> values = new ArrayList<V>();

        for (ArrayList<Pair> list : items) {
            if (list != null) {
                for (Pair pair : list) {
                    values.add(pair.Value());
                }
            }
        }
        return values;
    }

    public ArrayList<K> keys() {
        ArrayList<K> keys = new ArrayList<K>();

        for (ArrayList<Pair> list : items) {
            if (list != null) {
                for (Pair pair : list) {
                    keys.add(pair.Key());
                }
            }
        }
        return keys;
    }

    @Override
    public String toString() {
        String s = "";

        for (var list : items) {
            if (list != null) {
                for (Pair pair : list) {
                    s += pair.Key() + "|" + pair.Value() + " -> ";
                }
            }
        }
        s += "Count: " + size;
        return s;
    }

    public void update(K key, V value) {
        int hash = getHash(key);
        ArrayList<Pair> list = items[hash];

        if (list != null) {
            for (Pair pair : list) {
                if (pair.Key().equals(key)) {
                    pair.Value(value);
                    return;
                }
            }
        }

        if (list == null) {
            list = new ArrayList<>();
            items[hash] = list;
        }
        list.add(new Pair(key, value));
        size++;
    }

    private int calcPrimeLength(int length) {
        while (!isPrime(++length))
            ;
        return length;
    }

    private boolean isPrime(int number) {
        for (int i = 2; i <= number / 2; i++)
            if (number % i == 0)
                return false;
        return true;
    }

    private int getHash(K key) {
        return Math.abs(key.hashCode()) % items.length;
    }
}