import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.stream.Collectors;

public class OneR {
    private class Item {
        private String attribute;
        private String classification;
        private int frequency;

        public Item(String attribute, String classification) {
            this.attribute = attribute;
            this.classification = classification;
            this.frequency = 1;
        }

        public String attribute() {
            return attribute;
        }

        public String classification() {
            return classification;
        }

        public int frequency() {
            return frequency;
        }

        public void incrementFrequency() {
            frequency++;
        }

        @Override
        public boolean equals(Object obj) {
            Item item = (Item) obj;
            return attribute.equals(item.attribute()) &&
                    classification.equals(item.classification());
        }

        @Override
        public int hashCode() {
            return classification.hashCode() + attribute.hashCode();
        }
    }

    private class ItemSet {
        private String column;
        private Hashtable<String, Item> items;
        private double errorRate;

        public ItemSet(String column) {
            this.column = column;
            this.items = new Hashtable<String, Item>(10);
        }

        public Hashtable<String, Item> items() {
            return items;
        }

        public double errorRate() {
            return errorRate;
        }

        public void addItem(String attribute, String classification) {
            String key = attribute + "->" + classification;

            Item item = items.get(key);
            if (item == null) {
                item = new Item(attribute, classification);
                items.add(key, item);
            } else {
                item.incrementFrequency();
            }
        }

        public void process() {

            var result = items.values()
                    .stream()
                    .sorted(Comparator.comparing(Item::attribute).thenComparing(Item::frequency).reversed())
                    .collect(Collectors.toList());

            int total = 0;
            int correct = 0;
            String attribute = null;

            for (Item item : result) {
                total += item.frequency();

                String key = item.attribute() + "->" + item.classification();

                // Gruppenwechsel
                if (attribute == null || !attribute.equals(item.attribute())) {
                    attribute = item.attribute();
                    correct += item.frequency();
                } else {
                    items.remove(key);
                }
            }
            errorRate = 100.0 - (correct * 100.0 / total);
        }

        @Override
        public String toString() {

            String s = "";
            s += column + "\n";

            for (Item item : items.values())
                s += String.format("  %s->%s : %d\n", item.attribute, item.classification, item.frequency);

            if (!Double.isNaN(errorRate))
                s += String.format("\n  ErrorRate: %.2f %%\n", errorRate);

            return s;
        }
    }

    private Hashtable<String, String> solution;

    public OneR() {
        solution = new Hashtable<String, String>();
    }

    public void build(BufferedReader r) throws IOException {

        var list = new ArrayList<ItemSet>();

        String[] header = r.readLine().split(";");

        // Spaltenüberschriften lesen
        for (String head : header) {
            list.add(new ItemSet(head));
        }

        // Daten einlesen und Häufigkeiten ermitteln
        String line;

        while ((line = r.readLine()) != null) {
            String[] tokens = line.split(";");

            String classification = tokens[0];

            for (int i = 1; i < tokens.length; i++) {
                String attribute = tokens[i];

                ItemSet itemSet = list.get(i);
                itemSet.addItem(attribute, classification);
            }
        }
        r.close();

        // Gruppen mit bester Vorhersage pro Spalte ermitteln
        for (ItemSet items : list) {
            // vorher: Ausgabe alle Häufigkeitsverteilungen
            // System.out.println(items);
            items.process();
            // nacher: Ausgabe bester Häufigkeitsverteilungen
            // System.out.println(items);
        }

        // Spalte mit geringster Fehlerrate ermitteln
        ItemSet solutionSet = null;
        for (ItemSet items : list) {
            if (!Double.isNaN(items.errorRate())) {
                if (solutionSet == null || items.errorRate() < solutionSet.errorRate()) {
                    solutionSet = items;
                }
            }
        }

        // Lösung für Vorhersage speichern
        for (Item item : solutionSet.items().values())
            solution.add(item.attribute(), item.classification());

        System.out.println("_".repeat(80));
        System.out.println("Regeln: " + solutionSet);

        list.clear();
    }

    public String classify(String value) {
        return solution.get(value);
    }
}