public class RedBlackTree<K extends Comparable<K>, V> {
    private class Node {
        public K Key;
        public V Value;
        public boolean IsRed;
        public Node Left;
        public Node Right;

        public Node(K key, V value, boolean isRed) {
            Key = key;
            Value = value;
            IsRed = isRed;
        }
    }

    private Node root;
    private int size;

    public int size() {
        return size;
    }

    public K Min() {
        if (size == 0)
            throw new IllegalArgumentException("No items in tree.");

        Node node = root;

        K min = root.Key;

        while (node != null) {
            min = node.Key;
            node = node.Left;
        }
        return min;
    }

    public K Max() {
        if (size == 0)
            throw new IllegalArgumentException("No items in tree.");

        Node node = root;

        K max = root.Key;

        while (node != null) {
            max = node.Key;
            node = node.Right;
        }
        return max;
    }

    public TraverseModeEnum TraverseMode;

    public RedBlackTree() {
        TraverseMode = TraverseModeEnum.PreOrder;
    }

    public void add(K key, V value) {

        root = addTo(root, key, value);
        root.IsRed = false;

        size++;
    }

    private Node addTo(Node node, K key, V value) {

        if (node == null)
            return new Node(key, value, true);

        int cmp = key.compareTo(node.Key);
        if (cmp < 0)
            node.Left = addTo(node.Left, key, value);
        else if (cmp > 0)
            node.Right = addTo(node.Right, key, value);
        else
            node.Value = value;

        if (isRed(node.Right) && !isRed(node.Left)) // Regel für Linkslastigkeit
            node = rotateLeft(node);

        if (isRed(node.Left) && isRed(node.Left.Left)) // Keine aufeinanderfolgenden roten Knoten
            node = rotateRight(node);

        if (isRed(node.Left) && isRed(node.Right)) // keine zwei roten Kindknoten
            flipColors(node);

        return node;
    }

    private Node rotateRight(Node node) {
        Node newParent = node.Left;

        node.Left = newParent.Right;
        newParent.Right = node;
        newParent.IsRed = newParent.Right.IsRed;
        newParent.Right.IsRed = true;

        return newParent;
    }

    private Node rotateLeft(Node node) {
        Node newParent = node.Right;

        node.Right = newParent.Left;
        newParent.Left = node;
        newParent.IsRed = newParent.Left.IsRed;
        newParent.Left.IsRed = true;

        return newParent;
    }

    private void flipColors(Node node) {
        node.IsRed = !node.IsRed;
        node.Left.IsRed = !node.IsRed;
        node.Right.IsRed = !node.IsRed;
    }

    private boolean isRed(Node node) {
        if (node == null) // Null-Verweise stellen schwarze Blätter dar
            return false;

        return node.IsRed;
    }

    public boolean contains(K key) {
        Node node = root;

        while (node != null) {
            int c = key.compareTo(node.Key);

            if (c == 0)
                return true;

            if (c < 0)
                node = node.Left;
            else
                node = node.Right;
        }
        return false;
    }

    public V getValue(K key) {
        Node node = root;

        while (node != null) {
            int c = key.compareTo(node.Key);

            if (c == 0)
                return node.Value;

            if (c < 0)
                node = node.Left;
            else
                node = node.Right;
        }
        throw new IllegalArgumentException("Key doesn't exist.");
    }

    public void clear() {
        root = null;
        size = 0;
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder();
        int level = 0;

        traverse(root, level, s);

        return s.toString();
    }

    private void traverse(Node node, int level, StringBuilder s) {
        if (node == null)
            return;

        if (TraverseMode == TraverseModeEnum.PreOrder) {
            s.append(" ".repeat(level)).append(node.Key).append("\n");
            traverse(node.Left, level + 2, s);
            traverse(node.Right, level + 2, s);
        }
        if (TraverseMode == TraverseModeEnum.PostOrder) {
            traverse(node.Left, level + 2, s);
            traverse(node.Right, level + 2, s);
            s.append(" ".repeat(level)).append(node.Key).append("\n");
        }
        if (TraverseMode == TraverseModeEnum.InOrder) {
            traverse(node.Left, level + 2, s);
            s.append(" ".repeat(level)).append(node.Key).append("\n");
            traverse(node.Right, level + 2, s);
        }
        if (TraverseMode == TraverseModeEnum.ReverseInOrder) {
            traverse(node.Right, level + 2, s);
            s.append(" ".repeat(level)).append(node.Key).append("\n");
            traverse(node.Left, level + 2, s);
        }
    }
}